#include <libkmod.h>
#include <stdlib.h>

#include "control/job_scheduler.h"
#include "control/kmodule_fsm.h"
#include "control/kmodules_loader.h"
#include "control/worker_thread_set.h"

#include "util/logger.h"

static error_code_t kmodule_fsm_process_signal_schedule_job_in_state_added(kmodule_t *kmodule);
static error_code_t kmodule_fsm_process_signal_schedule_job_in_state_ready_to_schedule(kmodule_t *kmodule);

static error_code_t kmodule_fsm_enter_state_loading(kmodule_t *kmodule);
static void kmodule_fsm_enter_state_done(kmodule_t *kmodule);

static void kmodule_fsm_loading_done_callback(kmodule_t *kmodule, error_code_t ret_val);

//------------------------------------------- API member functions --------------------------------------------------
error_code_t kmodule_fsm_signal_schedule_job(kmodule_t *kmodule)
{
	kmodule_state_t state;
	state=kmodule_get_state(kmodule);

	if (state==ADDED)
		return kmodule_fsm_process_signal_schedule_job_in_state_added(kmodule);
	else if (state==READY_TO_SCHEDULE)
		return kmodule_fsm_process_signal_schedule_job_in_state_ready_to_schedule(kmodule);

	//ignore the signal when we are in any other state
	return RESULT_OK;
}

error_code_t kmodule_fsm_signal_depmod_loaded(kmodule_t *kmodule)
{
	kmodule_signal_depmod_loaded(kmodule);
	if (kmodule_all_depmods_loaded(kmodule))
		kmodule_set_state_ready_to_schedule(kmodule);

	return RESULT_OK;
}
//-------------------------------------------------------------------------------------------------------------------

//------------------------------------------- private member functions ----------------------------------------------
static error_code_t kmodule_fsm_process_signal_schedule_job_in_state_added(kmodule_t *kmodule)
{
	int deps_scheduled_cnt;
	error_code_t result;

	logger_log_debug("KMODULE_FSM -> module in state ADDED. Scheduling dependencies.");

	//schedule dependencies if needed
	result=job_scheduler_schedule_depmods(kmodule, &deps_scheduled_cnt);

	//change the state of the module
	if (result==RESULT_OK)
	{
		if (deps_scheduled_cnt==0)
		{
			logger_log_debug("KMODULE_FSM -> no open dependencies.");

			//all dependencies fine -> directly start loading this module if we are not busy,
			//otherwise mark it as ready for loading
			if (worker_thread_set_can_schedule_something())
				result=kmodule_fsm_enter_state_loading(kmodule);
			else
			{
				logger_log_debug("KMODULE_FSM -> No threads available at the moment to load the module.");
				kmodule_set_state_ready_to_schedule(kmodule);
			}
		}
		else
		{
			//we did schedule some dependencies -> enter state DEPMOD_SCHEDULED
			kmodule_set_state_depmod_scheduled(kmodule,deps_scheduled_cnt);
			logger_log_debug("KMODULE_FSM -> Scheduled %d kmodules to resolve dependencies.",deps_scheduled_cnt);
		}
	}

	return result;
}

static error_code_t kmodule_fsm_process_signal_schedule_job_in_state_ready_to_schedule(kmodule_t *kmodule)
{
	error_code_t result=RESULT_OK;

	if (worker_thread_set_can_schedule_something())
		result=kmodule_fsm_enter_state_loading(kmodule);

	return result;
}

static error_code_t kmodule_fsm_enter_state_loading(kmodule_t *kmodule)
{
	error_code_t result;
	result=worker_thread_set_start_loading_module(kmodule, kmodule_fsm_loading_done_callback);
	if (result==RESULT_OK)
		kmodule_set_state_loading(kmodule);
	return result;
}

static void kmodule_fsm_enter_state_done(kmodule_t *kmodule)
{
	kmodule_signal_done_to_waiting_modules(kmodule);
	//remove the module from the job schedulers list
	kmodule_remove_from_main_list(kmodule);
}

static void kmodule_fsm_loading_done_callback(kmodule_t *kmodule, error_code_t ret_val)
{
	const char *name=kmodule_get_name(kmodule);
	logger_log_debug("KMODULE_FSM -> Module %s processed by worker thread. Error code: %d",
			name,ret_val);
	if (ret_val==RESULT_OK)
		kmodule_fsm_enter_state_done(kmodule);
	else
		logger_log_error("Unable to load the kernel module %s.",name);
	kmodule_free(kmodule);
}
//-------------------------------------------------------------------------------------------------------------------
